/** @file   flameralien.cpp
 * @brief   Implementation of FlamerAlien - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "flameralien.h"
#include "flamerai.h"
#include "settings.h"
#include "warglobals.h"
#include "animplayer.h"
#include "soundsamples.h"
#include "AnimId.h"
using namespace eng2d;

namespace WeWantWar {

///
/// Static members, constants and datatypes
/// =======================================

const int FlamerAlien::GUN_CONTROLPOINT_INDEX;


///
/// Constructors, destructor and operators
/// ======================================
/** Constructor
 */
FlamerAlien::FlamerAlien() :
  Alien()
{
  int oid = ObjectID::TYPE_FLAMERALIEN;
  
  // Set default properties
  iWalkVec.vy = - Settings::floatObjProp(oid, "spd_walk:");
  this->setCorrectAnimation( GameAnims::EIdle );
  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  this->setArmor( Settings::floatObjProp(oid, "armor:") );
  
  // Set the collision points
  this->setCollisionPoint( 0, Vec2D( -13,-13 ) );
  this->setCollisionPoint( 1, Vec2D(  13,-13 ) );
  this->setCollisionPoint( 2, Vec2D(  13, 13 ) );
  this->setCollisionPoint( 3, Vec2D( -13, 13 ) );
  
  // Add the controlpoint for the weapon nose
  this->addCtrlPoint( Vec2D( 2, -25 ) );
  
  // Create the ai-object.
  this->setController( new FlamerAi( this ) );
}



/** Destructor
 */
FlamerAlien::~FlamerAlien()
{
}




///
/// Public methods
/// ==============


/** Updates this object
 */
void FlamerAlien::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  if ( this->state() == GameObject::STATE_DYING ) {
    if ( m_animation.paused() == true ) {
      // The dying animation has been played through. We mark this alien
      // as killed.
      this->state( GameObject::STATE_KILLED );
      this->setCorrectAnimation( GameAnims::EKilled );
    }
    return;
  }
  
  m_pController->update();  
  if ( m_pController->forward() != 0 ) {
    // We're walking
    Vec2D mvec( iWalkVec );
    mvec.rotate( this->angle() );
    this->move( mvec );
    this->setCorrectAnimation( GameAnims::EWalk );
  }
  
  this->changeAngle( m_pController->turn() );
  
  if ( m_pController->shoot() != 0 ) {
    // We should shoot
    this->attack();
  }
  
  if ( m_pController->forward() == 0 && m_pController->shoot() == 0 ) {
    this->setCorrectAnimation( GameAnims::EIdle );
  }
}


/** Makes sound
 */
void FlamerAlien::makeSound( GameObject::SoundID aSoundId ) const
{
  if ( aSoundId == GameObject::SND_ATTACK ) {
    Sound::playSample( SMP_WEAPON3, false );
  }
  if ( aSoundId == GameObject::SND_PAIN ) {
    Sound::playSample( SMP_FLAMERALIEN_SCREAM, false );
  }
  if ( aSoundId == GameObject::SND_DIE ) {
    Sound::playSample( SMP_ALIEN1_DIE, false );
  }
}



/** Handles the bullet hits
 */

bool FlamerAlien::hitByBullet( Bullet* aBullet )
{
  bool ret = Alien::hitByBullet( aBullet );
  if ( ret == true ) {
    if ( aBullet->iType != Bullet::EFlameThrower ) {
      ParticleBlood* pP = new ParticleBlood( aBullet->iPosition,
                                             aBullet->velocity(), 12,
                                             Color(130,60,10) );
      WarGlobals::pPartManager->addSystem( pP );
    }
  }

  // Inform the AI-controller that we got hit.
  AIController* pC = dynamic_cast<AIController*>( this->getController() );
  AIController::BFlag f = pC->getFlags();
  f &= ~AIController::WAIT_TARGET;
  pC->setFlags( f );
  return ret;
}


/** Kills this alien
 */
void FlamerAlien::kill()
{
  this->state( GameObject::STATE_DYING );
  this->setCorrectAnimation( GameAnims::EDying );
  this->makeSound( GameObject::SND_DIE );
}



///
/// Getter methods
/// ==============

/** Returns the type of this object
 */
ObjectID::Type FlamerAlien::objectType() const
{
  return ObjectID::TYPE_FLAMERALIEN;
}



///
/// Private or Protected methods
/// ============================

/** Sets the requested animation
 */
void FlamerAlien::setCorrectAnimation( int aAnimId )
{
  if ( this->getAnimID() != aAnimId ) {
    const Animation& anim = GameAnims::findAnimation( AnimId::KFlamerAlien, aAnimId );
    this->setAnimation( anim, aAnimId );
  }
}



/** Shoots
 */
void FlamerAlien::attack()
{
  if ( this->reloading() == false ) {
    // Handle shooting

    Vec2D gunV = this->getCtrlPoint( GUN_CONTROLPOINT_INDEX );
    gunV += this->position();

    Weapon::Specs& w = Weapon::getWeaponSpecs( Weapon::W_FLAMETHROWER );
    Bullet* pB = BulletTable::createBullet( this, gunV, Bullet::EFlameThrower );
    pB->iDamage /= 2;
    WarGlobals::pBulletManager->spawnBullet( pB );

    this->setCounter( RELOAD_COUNTER_INDEX, w.reload );
    this->makeSound( GameObject::SND_ATTACK );
  }
  this->setCorrectAnimation( GameAnims::EShoot );
}

} // end of namespace
